/* Simulate accumulation and removal by selection of deleterious mutations.  
Infinite alleles model; only the number of mutations (all assumed to have same
selection coeff) is stored.  Population size is fixed, and is always 1/2 male, 1/2
female.  Each pair of parents have on average noff_mean  offspring; at most popsize 
offspring from the entire collection survive to reproduce in the next generation.  
Survival is weighted by fitness, based on multiplicative selection. When soft selection 
is chosen, survival probability is relative to that of most fit extant class, 
not theoretical maximum. */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include "prano.h"
#include "brano.h"
#include "mtwist.h"
int multinomial(int n, int nclass, double p[], int rv[]);

int main(int argc, char **argv) {
  const int fixedOff = 0;
  int doHard = 1;
  const int doTrunc = 0;
  const double truncThresh = .932;
  double s = .01;
  double ndelet = 1.1;
  double nexp, fitness;
  int *indiv, *offspring; // stores number of mutations carried 
  int mut_size=10000;   // max number of mutations carried by individual (increases automatically)
  int *slots, index, popsize;
  int ngen = 100000, ind, ioff, igen, nind, noff_mean;
  int noff;  // noff_mean is mean offspring per pair; noff = actual number for a breeding pair 
  int nmale, totoff, offbuffsize;
  int maxMut, minMut, sumMut, left;
  int *offWithNmut, ifit, *found;
  double *expected, norm, nsurvive, mostfit;


  if (argc != 6) {
    fprintf(stderr, "Usage: deleterious <sel coeff> <N mut/gen> <pop size> <N offspring/couple> <hard selection?>\n");
    exit(0);
  }
  doHard = atoi(argv[5]);
  noff_mean = atof(argv[4]);
  popsize = atoi(argv[3]);
  s = atof(argv[1]);
  ndelet = atof(argv[2]);
  indiv = malloc(popsize * sizeof(int));
  slots = malloc(popsize * sizeof(int));
  offWithNmut = malloc(mut_size * sizeof(int));
  found = malloc(mut_size * sizeof(int));
  expected = malloc(mut_size * sizeof(double));

  offbuffsize = noff_mean * popsize / 2;
  offspring = malloc(offbuffsize * sizeof(int));
  assert(offspring != NULL);
  nind = popsize;
  for (ind = 0; ind < popsize; ind++) {indiv[ind] = 0;}
  mt_seed();
  fprintf(stderr, "Generation\tmin mutations\tmax mutations\tmean N mutations\tN offspring\n");

  for (igen = 0; igen < ngen; igen++) {
    if (nind%2 == 1) {nind--;}
    nmale = nind / 2;
    totoff = 0;

    /* Create offspring from pairs of individuals */
    for (ind = 0; ind < nmale; ind++) {
      // If number of offspring is variable, pick. 
      if (fixedOff != 1) {
	noff = prano(noff_mean);
      }
      /* mean number of mutations carried by offspring of these parents */
      nexp = .5 * (indiv[ind] + indiv[ind+nmale]) + ndelet;  
      for (ioff = 0; ioff < noff; ioff++) {
	/* Poisson number of new mutations in this child + binomial inherited */
	if (totoff >= offbuffsize) {
	  offbuffsize *= 2;
	  offspring = realloc(offspring, offbuffsize * sizeof(int));
	}
	offspring[totoff] = brano(.5, indiv[ind]) + brano(.5, indiv[ind+nmale]) + prano(ndelet);
	totoff++;
      }
    }
    if (fixedOff == 1) {assert( totoff == (int) (noff_mean * nind / 2 + .5) ); }

    /* Tabulate the number of offspring with each number of mutants. */
    for (ifit = 0; ifit < mut_size; ifit++) {offWithNmut[ifit] = 0;}
    for (ioff = 0; ioff < totoff; ioff++) {
      if (offspring[ioff] >= mut_size) {
	mut_size *= 2;
	offWithNmut = realloc(offWithNmut, mut_size * sizeof(int));
	found = realloc(found, mut_size * sizeof(int));
	expected = realloc(expected, mut_size * sizeof(double));
      }
      assert(offspring[ioff] < mut_size);
      offWithNmut[offspring[ioff]]++;
    }
    
    /* Draw the members of the next generation, based on the number of offspring 
       with each number of mutants, weighted by their fitness.  */
    norm = 0;
    fitness = 1;
    if (doHard == 0) {
      mostfit = -1;
      for (ifit = 0; ifit < mut_size; ifit++) {
	if (mostfit < 0 && offWithNmut[ifit] > 0) {
	  mostfit = fitness;
	  break;
	}
	fitness *= (1-s);
      }
    }
    else {
      mostfit = 1;
    }
    fitness = 1;
    nsurvive = 0;  
    for (ifit = 0; ifit < mut_size; ifit++) {
      norm += offWithNmut[ifit] * fitness / mostfit;
      expected[ifit] = fitness * offWithNmut[ifit] / mostfit;
      nsurvive += expected[ifit];
      fitness *= (1-s);
      if (doTrunc && fitness < truncThresh) {fitness = 0;}
    }
    for (ifit = 0; ifit < mut_size; ifit++) {
      expected[ifit] /= norm;
    }
    nind = (nsurvive < popsize) ? (int) (nsurvive+.5) : popsize;
    if (nind < 2) {
      fprintf(stderr, "Extinction.\n");
      exit(0);
    }
    multinomial(nind, mut_size, expected, found);

    /* found[] holds the number individuals in the next generation for 
       each fitness class.  Distribute the individuals to random slots 
       in the next generation. */
    for (ind = 0; ind < nind; ind++) {slots[ind] = ind;}
    left = nind;
    for (ifit = 0; ifit < mut_size; ifit++) {
      for (ind = 0; ind < found[ifit]; ind++) {
	/* Pick a random place, from the list of available slots, to stick the individual */
	index = (int) (mt_drand() * left);
	assert(index < left);
	indiv[slots[index]] = ifit;
	slots[index] = slots[left];
	left--;
      }
    }

    maxMut = 0;
    minMut = 99999;
    sumMut = 0;
    for (ind = 0; ind < nind; ind++) {
      sumMut += indiv[ind];
      if (indiv[ind] > maxMut) {maxMut = indiv[ind];}
      if (indiv[ind] < minMut) {minMut = indiv[ind];}
    }  
    if (igen % 10 == 0) {
      fprintf(stderr, "%d\t%d\t%d\t%.2f\t%.0f\n", igen, minMut, maxMut, 
	     (double) sumMut / nind, nsurvive);
    }
  }  
  return 0;
}

int multinomial(int n, int nclass, double p[], int rv[]) {
  /* Lifted from Dick Hudson */
  double x, s;
  int i, j=0;
  for(i=0; i<nclass; i++) rv[i]=0;
  for(i=0; i<n ; i++) {
    x = mt_drand();
    j=0;
    s = p[0];
    while( (x > s) && ( j<(nclass-1) ) )  s += p[++j];
    rv[j]++;
  }
  return(j);
}
