/*
  The Broad Institute
  SOFTWARE COPYRIGHT NOTICE AGREEMENT
  This software and its documentation are copyright (2005) by the
  Broad Institute/Massachusetts Institute of Technology. All rights are
  reserved.

  This software is supplied without any warranty or guaranteed support
  whatsoever. Neither the Broad Institute nor MIT can be responsible for its
  use, misuse, or functionality.
*/
/* $Id: simulator.c,v 1.12 2006/01/18 19:26:26 sfs Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include "defs.h"
#include "segment.h"
#include "node.h"
#include "nodelist.h"
#include "pop.h"
#include "demography.h"
#include "../cosi_rand/random.h"
#include "historical.h"
#include "migrate.h"
#include "recomb.h"
#include "geneconversion.h"
#include "haplos.h"
#include "mutlist.h"
#include "mutate.h"
#include "coalesce.h"
#include "../cosi_rand/multinom.h"
#include "../cosi_rand/poisson.h"

double sim_get_poisson_rate(void);
int sim_do_poisson (genid /* gen */);
int sim_complete (void);
genid sim_get_hist_event (genid /* gen */);
double sim_get_pois_event (void);

static double coalesce_rate;
static double migrate_rate;
static double recombination_rate;
static double geneconv_rate;
static double poisson_rate;
static double theta;
static int fixed_num_mut = -1;


void 
sim_set_num_muts(int nmut) {fixed_num_mut = nmut;}


/* To execute the coalescent simulator:
 * 1. Calculate the times to the next historical and poisson events.
 * 2. Choose the closest event, and execute it.
 * 3. Repeat until sim_complete().
 * 
 * Important assumption:
 * Historical events are appropriately spaced such that
 * historical_event_time will never be negative unless
 * there are no historical events left.
 *
 */
genid
sim_execute (void) 
{
	genid gen = 0;
	genid historical_event_time;
	genid poisson_event_time;	
	int coal = 0;
	int complete_flag = 0;	

	while (complete_flag == 0) {
	  coal = 0;
	  historical_event_time = (double) sim_get_hist_event (gen);
	  poisson_event_time = sim_get_pois_event ();
	  
	  if (DEBUG) {
	    printf("recomb: %f coalesce: %f migrate: %f geneconvert: %f\n", 
		   recombination_rate, coalesce_rate, migrate_rate, geneconv_rate);
	    printf("poisson time: %f, hist time: %f\n", 
		   poisson_event_time, historical_event_time);
	  }
	  
	  if (historical_event_time < 0 
	      || poisson_event_time < historical_event_time) {
	    gen += poisson_event_time;
	    coal = sim_do_poisson (gen);
	    if (coal) {complete_flag = dg_done_coalescent();}
	  }
	  else {
	    gen += historical_event_time;
	    gen = historical_event_execute(gen);
	    complete_flag = dg_done_coalescent(); 
	  }

	}
	return gen;
}

/* This is called after sim_execute is finished. */
/*
 * A region is defined as the sequence between two recombination
 * events.
 *
 * 1. Calculate the total time in the tree for all the branches for
 *    all regions.
 * 2. Mutations are poisson distributed, with lambda specified by
 *    the mutation rate. Each region has a number of mutations 
 *    proportional to the time spent in the tree for that region.
 * 3. For each mutation in that region, choose a random location
 *    within that region, and call a function that places the
 *    mutation.
 */
int 
sim_mutate (FILE *fp, Mutlist *muts, Haplos *haps) 
{
  int i, summut = 0;
  double mutrate, prob_sum=0;
  double loc;
  int numregions, reg, nummuts=0;
  double randmark;
  double begin, *reglen=NULL;
  double *prob_region=NULL, *treetime=NULL;
  int *nmut_by_region=NULL;
  
  numregions = dg_get_num_regs ();
  
  mutate_print_headers(fp);
  
  reglen = malloc(numregions * sizeof(double));
  treetime = malloc(numregions * sizeof(double));
  for (reg = 0; reg < numregions; reg++) {
    treetime[reg] = dg_total_tree_time (reg);
    reglen[reg] = dg_get_reg_length(reg);
    prob_sum += treetime[reg] * reglen[reg];
  }
  if (fixed_num_mut > 0) {
    prob_region = calloc(numregions, sizeof(double));
    nmut_by_region = malloc(numregions * sizeof(int));
    for (reg = 0; reg < numregions; reg++) {
      prob_region[reg] = treetime[reg] *reglen[reg] / prob_sum;
    }
    multinom(numregions, fixed_num_mut, prob_region, nmut_by_region);
  }

  for (reg = 0; reg < numregions; reg++) {
    begin = dg_reg_begin(reg);
    
    if (fixed_num_mut == -1) {
      mutrate = theta * treetime[reg] * reglen[reg];
      nummuts = poisson(mutrate);
      if (fp != NULL) {fprintf(fp, "> [%f, %f]  time %d  E[muts] = %f (%d)\n", 
			       begin, reglen[reg], (int) treetime[reg], mutrate, nummuts);}
    }
    else {
      nummuts = nmut_by_region[reg];
      if (fp != NULL) {fprintf(fp, "> [%f, %f]  time %d  E[muts] = %f (%d)\n", 
			       begin, reglen[reg], (int) treetime[reg], 0., nummuts);}
    }
    summut += nummuts;
    
    for (i = 0; i < nummuts; i++) {
      loc = begin + random_double() * reglen[reg];
      randmark = random_double();
      mutate_find_and_print(fp, reg, loc, randmark, 
			    treetime[reg], muts, haps);
    }
  }
  free(reglen);
  free(prob_region);
  free(treetime);
  free(nmut_by_region);
  return nummuts;	  
}

void
sim_settheta (double t) 
{
	theta = t;
}

void
sim_setlength (int l) 
{
	recomb_set_length (l);
	gc_set_length (l);
}

/*****************************************************************/
/* INTERNAL FUNCTIONS */

genid 
sim_get_hist_event (double gen) 
{
	return historical_get_next (gen);
}

double 
sim_get_pois_event (void) 
{
	return poisson_get_next (sim_get_poisson_rate());
}

double 
sim_get_poisson_rate(void) 
{
	coalesce_rate = coalesce_get_rate();
	migrate_rate = migrate_get_rate();
	recombination_rate = recomb_get_rate();
	geneconv_rate = gc_get_rate();

	poisson_rate = (double) (coalesce_rate + migrate_rate + recombination_rate + geneconv_rate);
	return poisson_rate;
}

int 
sim_do_poisson (genid gen) 
{
  int did_coal = 0;
  double randdouble = random_double();
  double dum, dum2;
  int popindex;
  
  if (randdouble < recombination_rate / poisson_rate) {
    popindex = recomb_pick_popindex();
    recomb_execute(gen, popindex, &dum);
  }
  else if (randdouble < (recombination_rate + migrate_rate) / poisson_rate) {
    migrate_execute(gen);
  }
  else if (randdouble < (recombination_rate + migrate_rate + coalesce_rate) / poisson_rate) {
    popindex = coalesce_pick_popindex();
    dg_coalesce_by_index (popindex, gen);
    did_coal = 1;
  }
  else {
    popindex = gc_pick_popindex();
    gc_execute(gen, popindex, &dum, &dum2);
  }
  return did_coal;
}

int 
sim_complete (void) {
	return dg_done_coalescent();

}
