/*
  The Broad Institute
  SOFTWARE COPYRIGHT NOTICE AGREEMENT
  This software and its documentation are copyright (2005) by the
  Broad Institute/Massachusetts Institute of Technology. All rights are
  reserved.

  This software is supplied without any warranty or guaranteed support
  whatsoever. Neither the Broad Institute nor MIT can be responsible for its
  use, misuse, or functionality.
*/
/* $Id: recomb.c,v 1.6 2005/09/22 18:32:18 sfs Exp $ */
#include <stdlib.h>
#include <stdio.h>
#include "defs.h"
#include "../cosi_rand/random.h"
#include "segment.h"
#include "node.h"
#include "nodelist.h"
#include "pop.h"
#include "demography.h"
#include "recomb.h"

double rec_recombrate = 0;
double rec_lastrate = 0;
int rec_length;

typedef struct recombstructure {
  int startbase;
  double rate;
  double cumulative;
  struct recombstructure *next;
} RecombStruct;

RecombStruct *recombs = NULL;

void 
recomb_add (int start, double rate)
{
  RecombStruct *newrecomb = malloc (sizeof (RecombStruct));
  RecombStruct *temprecomb = recombs;
	
  newrecomb->startbase = start;
  newrecomb->rate = rate;
	
  if (recombs == NULL) {
    recombs = newrecomb;
    newrecomb->next = NULL;
    return;
  }
	
  else if (start < recombs->startbase) {
    newrecomb->next = recombs;
    recombs = newrecomb;
    return;
  }

  else {
    while (temprecomb->next != NULL) {
      if (temprecomb->next->startbase > start) {
	newrecomb->next = temprecomb->next;
	temprecomb->next = newrecomb;
	return;
      }
      temprecomb = temprecomb->next;
    }

    newrecomb->next = NULL;
    temprecomb->next = newrecomb;
  }
	
}

void recomb_set_length (int l) {
  rec_length = l;
}


void recomb_calc_r() 
{
  double rr = 0;
  RecombStruct *temprecomb = recombs;

  while (temprecomb != NULL && temprecomb->startbase < rec_length) {
    if (temprecomb->next == NULL || 
	temprecomb->next->startbase > rec_length) {
      rr += (rec_length - temprecomb->startbase) * 
	temprecomb->rate;
    }
    else
      rr += (temprecomb->next->startbase - 
	     temprecomb->startbase) * temprecomb->rate;
		
    temprecomb->cumulative = rr;
    temprecomb = temprecomb->next;

  }
  rec_recombrate = rr;
}

double recomb_get_r(void) {
  return rec_recombrate;
}

double recomb_get_rate (void) {
  int numpops = dg_get_num_pops();
  int i;
  double rate = 0;
  int numnodes;
  if (rec_recombrate == 0) return 0;

  for (i = 0; i < numpops; i++) {
    numnodes = dg_get_num_nodes_in_pop_by_index (i);
    rate += ( numnodes * rec_recombrate);
  }

  rec_lastrate = rate;

  return rate;  

}

int recomb_pick_popindex(void) {
  /* figure out which pop to recombine */
  /* Note: the use of lastrate looks fragile to me, with little time savings.
     It assumes that get_rate has been called immediately before this 
     function, with no change in the number of chromosomes since. sfs */
  int popindex = 0, i, numnodes;
  double randcounter = random_double() * rec_lastrate;
  double rate = 0;
  int numpops = dg_get_num_pops();
  
  /* weigh pops by numnodes. */
  if (numpops > 1) {
    for (i = 0; i < numpops && rate < randcounter; i++) {
      numnodes = dg_get_num_nodes_in_pop_by_index(i);
      rate += (numnodes * rec_recombrate);
    }
    popindex = i - 1;
  }
  return popindex;
}


Node** recomb_execute (genid gen, int popindex, double *location) {
  
  /*  pick location */  
  double loc;
  double temp;
  double temp1;
  RecombStruct *temprecomb = recombs;
  double rr = 0;
  double end;
		       
  /* choosing location.. */
  temp1 = random_double();
  temp = (double) (temp1 * (rec_recombrate));

  while (temprecomb != NULL && temprecomb->startbase < rec_length && rr < temp) {
    if (temprecomb->next == NULL || 
	temprecomb->next->startbase > rec_length) {
      rr += (rec_length - temprecomb->startbase) * temprecomb->rate;
    }
    else {
      rr += (temprecomb->next->startbase - temprecomb->startbase) * 
	temprecomb->rate;			
    }

    if (rr < temp) {
      temprecomb = temprecomb->next;
    }
  }

  if (temprecomb->next == NULL || temprecomb->next->startbase > rec_length)
    end = rec_length;
  else end = temprecomb->next->startbase;

  loc = (double)((int) (end - (rr - temp) / temprecomb->rate)) / rec_length;
  *location = loc;

  return dg_recombine_by_index (popindex, gen, loc);
}
