/*
  The Broad Institute
  SOFTWARE COPYRIGHT NOTICE AGREEMENT
  This software and its documentation are copyright (2005) by the
  Broad Institute/Massachusetts Institute of Technology. All rights are
  reserved.

  This software is supplied without any warranty or guaranteed support
  whatsoever. Neither the Broad Institute nor MIT can be responsible for its
  use, misuse, or functionality.
*/
/* $Id: node.c,v 1.6 2006/01/18 19:26:25 sfs Exp $ */
/* node.c
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include "defs.h"
#include "segment.h"
#include "node.h"

int node_index = 0;

Node * make_empty_node_2 (genid /* gen */, int, Node *, Node *);
void node_exit(char*, char *);

void 
node_exit(char* funct_name, char* error_string) 
{
  fprintf(stderr, "node.c | %s: %s\n",
	  funct_name, error_string);
  exit(EXIT_FAILURE);
}

void
node_delete (Node* node) {
  if (node != NULL) {
    node_delete(node->descendent[0]);
    node_delete(node->descendent[1]);
    free(node);
    node = NULL;
  }
}

Node* 
make_new_node (double begin, double end, genid gen, popid pop) 
{
  Node *newnodeptr;
	
  if ((newnodeptr = (Node *)malloc (sizeof (*newnodeptr))) == NULL) {
    node_exit("make_new_node", "malloc failure");
  }
  newnodeptr->name = node_index;
  node_index++;
	
  newnodeptr->segs = NULL;
  newnodeptr->segs = segment_add (newnodeptr->segs, begin, end);
	
  newnodeptr->gen = gen;
  newnodeptr->pop = pop;
	
  newnodeptr->descendent[0] = NULL;
  newnodeptr->descendent[1] = NULL;

	
  return newnodeptr;
}

Node * 
make_empty_node (genid gen, popid pop, Node *des1) 
{
  Node *newnodeptr;

  if ((newnodeptr = (Node *)malloc (sizeof (*newnodeptr))) == NULL) {
    node_exit("make_empty_node", "malloc failure");
  }
	
  newnodeptr->name = node_index;
  node_index++;
	
  newnodeptr->gen = gen;
  newnodeptr->pop = pop;
  newnodeptr->descendent[0] = des1;
  newnodeptr->descendent[1] = NULL;

  return newnodeptr;
}

Node * 
node_coalesce (Node * node1, Node * node2, genid gen) 
{  
  Node * newnodeptr;
	
  newnodeptr = make_empty_node_2(gen, node1->pop, node1, node2);
  newnodeptr->segs = segment_union(node1->segs, node2->segs);  
  return newnodeptr;
}


int
node_recombine (Node *node, Node **newnode1, Node **newnode2, 
		genid gen, double loc) 
{
  Segment *seg, *seg1, *seg2;
	
  seg = segment_new (0, loc);
	
  seg1 = segment_intersect(node->segs, seg);
  seg2 = segment_intersect(node->segs, segment_inverse(seg));

  free(seg);
  seg = NULL;
	

  if (seg1 == NULL || seg2 == NULL) {
    if (seg1 != NULL) {
      free(seg1);
      seg1 = NULL;
    }
		
    if (seg2 != NULL) {
      free(seg2);
      seg2 = NULL;
    }
    return 1;
  }

  *newnode1 = make_empty_node (gen, node->pop, node);
  (*newnode1)->segs = seg1;
  *newnode2 = make_empty_node (gen, node->pop, node);
  (*newnode2)->segs = seg2;

  return 2;
}



/* GENE CONVERSION */
int
node_gc (Node *node, Node **newnode1, Node **newnode2, 
	 genid gen, double loc, double locend) 
{
  Segment *seg, *seg1, *seg2;
	
  seg = segment_new (loc, locend);
	
  seg1 = segment_intersect(node->segs, seg);
  seg2 = segment_intersect(node->segs, segment_inverse(seg));

  free(seg);
  seg = NULL;
	
  if (seg1 == NULL || seg2 == NULL) {
    if (seg1 != NULL) {
      free(seg1);
      seg1 = NULL;
    }
		
    if (seg2 != NULL) {
      free(seg2);
      seg2 = NULL;
    }
    return 1;
  }

  *newnode1 = make_empty_node (gen, node->pop, node);
  (*newnode1)->segs = seg1;
  *newnode2 = make_empty_node (gen, node->pop, node);
  (*newnode2)->segs = seg2;

  return 2;
}

int 
node_break_off_seg (Node *node, Node **newnode1, Node **newnode2, 
		    double begin, double end) 
{
  Segment *seg, *seg1, *seg2;
  genid gen = node->gen;

  seg = segment_new (begin, end);

  seg1 = segment_intersect(node->segs, seg);
  seg2 = segment_intersect(node->segs, segment_inverse(seg));

  free(seg);
  seg = NULL;
	
  if (seg1 == NULL || seg2 == NULL) {
    if (seg1 != NULL) {
      free(seg1);
      seg1 = NULL;
    }
		
    if (seg2 != NULL) {
      free(seg2);
      seg2 = NULL;
    }
    return 1;
  }

  *newnode1 = make_empty_node (gen, node->pop, node);
  (*newnode1)->segs = seg1;
  *newnode2 = make_empty_node (gen, node->pop, node);
  (*newnode2)->segs = seg2;

  return 2;

}

Node * 
make_empty_node_2 (genid gen, popid pop, Node *des1, Node *des2) 
{
  Node* newnodeptr;
	
  if ((newnodeptr = (Node *)malloc (sizeof (*newnodeptr))) == NULL) {
    node_exit("make_empty_node_2", "malloc failure");
  }
	
  newnodeptr->name = node_index;
  node_index++;
	
  newnodeptr->gen = gen;
  newnodeptr->pop = pop;
  newnodeptr->descendent[0] = des1;
  newnodeptr->descendent[1] = des2;
  newnodeptr->segs = NULL;
	
  return newnodeptr;
}

void 
print_node (Node* nodeptr) 
{
  int i = 0;
	
  printf("node: %d pop: %d gen: %f segs: ", nodeptr->name,
	 nodeptr->pop, nodeptr->gen);
  print_segs(nodeptr->segs);
  printf(" des: ");
  while (i < 2 && nodeptr->descendent[i] != NULL) {
    printf("%d ", nodeptr->descendent[i]->name);
    i++;
  }
  printf("\n");
}
