/*
  The Broad Institute
  SOFTWARE COPYRIGHT NOTICE AGREEMENT
  This software and its documentation are copyright (2005) by the
  Broad Institute/Massachusetts Institute of Technology. All rights are
  reserved.

  This software is supplied without any warranty or guaranteed support
  whatsoever. Neither the Broad Institute nor MIT can be responsible for its
  use, misuse, or functionality.
*/
/* $Id: mutate.c,v 1.7 2005/09/22 18:32:17 sfs Exp $ */
#include <stdlib.h>
#include <stdio.h>
#include "defs.h"
#include "segment.h"
#include "node.h"
#include "haplos.h"
#include "mutlist.h"
#include "mutate.h"
#include "nodelist.h"
#include "pop.h"
#include "demography.h"

typedef struct mutatednodelist {
  nodeid node;
  struct mutatednodelist *next;
} MutNodeList;


/* the mutation occurred between anc_node1 and anc_node2
   and appears in the nodes listed in mutatednodelist */
typedef struct mutations {  
  double location;
  Node* anc_node1;
  Node* anc_node2;
  MutNodeList *mutnodes;
} Mutations;

Mutations * mutate_region (Node * headnode, double loc, 
			   genid tree_age, double randmark);
Node * mutate_recurse (Node * nodeptr, double * ratio, 
		       Mutations * newmuts, genid tree_age,
		       double threshold, double loc);
MutNodeList * mutate_descendents (Node * tempnode, 
				  MutNodeList * mnlist, 
				  double loc);
MutNodeList * mutate_add_desc (MutNodeList * newmnl, 
			       MutNodeList * mnlist);
void mutate_print (FILE* fileptr, Mutations * mutptr, Mutlist *mutlist, Haplos *chromlist);
void mutate_print_haps (FILE* fileptr,  Mutations *mutptr);

/*  MUTATE_FIND_AND_PRINT
 *  this can be called once per mutation
 *  or it can edited to be called once per region, with a list of mutated
 *  sites.
 */
void 
mutate_find_and_print (FILE * fileptr, int regionnum, double loc, 
		       double randmark, genid tree_time, Mutlist *mutlist, 
		       Haplos *haps) 
{
  Mutations *muts;
  Node *nodeptr;
  
  nodeptr = dg_get_head_node (regionnum);
  muts = mutate_region(nodeptr, loc, tree_time, randmark);
  
  if (MUTATE_DEBUG)
    printf("mark: %f ", randmark);
  mutate_print(fileptr, muts, mutlist, haps);
  
}

void 
mutate_print_headers (FILE *outputptr) 
{
  if (outputptr != NULL) {fprintf(outputptr, "  pos \tanc1 anc2 \tfreq \tnodes...\n");}
}

/***********************************************************/
/* INTERNAL FUNCTIONS
 */

Mutations * 
mutate_region (Node * headnode, double loc,
	       genid tree_age, double randmark) 
{
  
  double ratio = 0;
  Mutations *newmuts;
  Node *mutatenode, *tempnode;
  
  tempnode = headnode;
  
  newmuts = (Mutations *) malloc (sizeof (*newmuts));
  newmuts->location = loc;
  
  mutatenode = mutate_recurse (tempnode, &ratio, newmuts, 
			       tree_age, randmark, loc);

  newmuts->mutnodes = mutate_descendents (mutatenode, NULL, loc);
  return newmuts;
}

/* HOW MUTATION WORKS RECURSIVELY */
/*
 * 1. Check the left descendent. If the left exists, continue.
 *    If not, return NULL.
 * 2. If it contains a relevant node
 *    (i.e. the node is active at the point of mutation)
 *    then continue. Else go to Step 4.
 * 3. Is this the randomly-selected, weighted-by-time branch?
 *    If so, put the mutation there. If not, recurse on this
 *    descendent.
 * 4. If the RIGHT descendent contains a relevant node then continue.
 *    Else return NULL.
 * 5. Is this the randomly-selected, weighted-by-time branch?
 *    If so, put the mutation there. If not, recurse on this
 *    descendent.
 */
Node * 
mutate_recurse (Node * nodeptr, double *ratioptr, Mutations *newmuts, 
		genid tree_age, double threshold, double loc)
{
  Node *mutatenode = NULL;
	
  if (nodeptr->descendent[0] != NULL) {
    if (seg_contains(nodeptr->descendent[0]->segs, loc)) {
			
      *ratioptr += (double) 
	(nodeptr->gen - nodeptr->descendent[0]->gen)
	/ tree_age;
			
      if (*ratioptr > threshold) {
	newmuts->anc_node1 = nodeptr;
	newmuts->anc_node2 = nodeptr->descendent[0];
	return nodeptr->descendent[0];
      }
			
      mutatenode = mutate_recurse (nodeptr->descendent[0], 
				   ratioptr, 
				   newmuts, tree_age, 
				   threshold, loc);
      if (mutatenode != NULL) return mutatenode;
      /* this happens if we place a mutation inside this recursion. */
    }
  }	
  else return NULL;


  if ((nodeptr->descendent[1] != NULL) &&
      (seg_contains(nodeptr->descendent[1]->segs, loc))) {
    *ratioptr += (double) (nodeptr->gen 
			   - nodeptr->descendent[1]->gen)
      / tree_age;
		
    if (*ratioptr > threshold) {
      newmuts->anc_node1 = nodeptr;
      newmuts->anc_node2 = nodeptr->descendent[1];
      return nodeptr->descendent[1];
    }
    mutatenode = mutate_recurse (nodeptr->descendent[1], 
				 ratioptr,
				 newmuts, tree_age, 
				 threshold, loc);
    if (mutatenode != NULL) return mutatenode;
    /* this happens if we place a mutation inside this recursion. */
		
  }
  return NULL;
}
	
/* mutate_descendents
 * also inserts them in numerical order.
 */
MutNodeList * 
mutate_descendents (Node * tempnode, MutNodeList *mnlist, double loc) 
{
  MutNodeList *newmnlist = NULL;

  /* if we are at a bottom node, add this node name. */
  if (tempnode->descendent[0] == NULL) {
    newmnlist = (MutNodeList *) malloc (sizeof (*newmnlist));
		
    newmnlist->node = tempnode->name;
    newmnlist->next = NULL;
		
    return mutate_add_desc(newmnlist, mnlist);
  }
  else {
    /* if the 0 descendent has this loc, traverse it. */
    if (seg_contains(tempnode->descendent[0]->segs, loc)) {
      newmnlist = 
	mutate_descendents(tempnode->descendent[0],
			   mnlist, loc);
      mnlist = newmnlist;
    }
    if (tempnode->descendent[1] != NULL)
      /* if the 1 descendent has this loc, traverse it. */
      if (seg_contains(tempnode->descendent[1]->segs, 
		       loc))
	newmnlist = 
	  mutate_descendents(tempnode->descendent[1], 
			     mnlist, loc);
    return newmnlist;
		
  }
}

MutNodeList * 
mutate_add_desc (MutNodeList *newmnl, MutNodeList *mnlist) 
{
  if (mnlist == NULL) {
    newmnl->next = mnlist;
    return newmnl;
  }
  else if (mnlist->node > newmnl->node) {
    newmnl->next = mnlist;
    return newmnl;
  }
  else {
    mnlist->next = mutate_add_desc(newmnl, mnlist->next);
    return mnlist;
  }
}


void 
mutate_print (FILE *outputptr, Mutations *mutptr, Mutlist *mutlist, Haplos *haps) 
{
  int i = 0, ichr;
  MutNodeList *tempmnlist;

  if (outputptr != NULL) {
    fprintf(outputptr, "M %.8f \t%d %d \t", mutptr->location,
	    mutptr->anc_node1->name, mutptr->anc_node2->name);
  }
	
  tempmnlist = mutptr->mutnodes;
	
  while (tempmnlist != NULL) {
    tempmnlist = tempmnlist->next;
    i++;
  }
	
  if (outputptr != NULL) 	fprintf(outputptr, "%d  \t", i);
	
  tempmnlist = mutptr->mutnodes;
	
  while (tempmnlist != NULL) {
    if (outputptr != NULL) fprintf(outputptr, "%d ", tempmnlist->node);
    /*	  fprintf(stderr, "muts: %d\n", tempmnlist->node); */
    ichr = tempmnlist->node;
    if (haps->nmut[ichr] == haps->mut_array_size[ichr]) {
      haps->mut_array_size[ichr] *= 2;
      haps->mutindex[ichr] = realloc(haps->mutindex[ichr], haps->mut_array_size[ichr]*sizeof(int));
    }
    /*	  printf("ichr: %d  nmut: %d\n", ichr, haps->nmut[ichr]); */
    haps->mutindex[ichr][haps->nmut[ichr]] = mutlist->nmut;
    haps->nmut[ichr]++;
    tempmnlist = tempmnlist->next;
  }

  if (mutlist->nmut == mutlist->array_size) {
    mutlist->array_size *= 2;
    mutlist->pos = realloc(mutlist->pos, mutlist->array_size * sizeof(double));
  }
  mutlist->pos[mutlist->nmut] = mutptr->location;
  mutlist->nmut++;
	
  if (outputptr != NULL) fprintf(outputptr, "\n");
}

