/*
  The Broad Institute
  SOFTWARE COPYRIGHT NOTICE AGREEMENT
  This software and its documentation are copyright (2005) by the
  Broad Institute/Massachusetts Institute of Technology. All rights are
  reserved.

  This software is supplied without any warranty or guaranteed support
  whatsoever. Neither the Broad Institute nor MIT can be responsible for its
  use, misuse, or functionality.
*/
/* $Id: migrate.c,v 1.6 2005/09/22 18:32:17 sfs Exp $ */

/* MIGRATE.C
 * determines when a migration should next occur.
 *
 * c.f. migrate.h
 */

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include "defs.h"
#include "segment.h"
#include "node.h"
#include "nodelist.h"
#include "pop.h"
#include "demography.h"
#include "../cosi_rand/random.h"

typedef struct migrate_struct {
  popid frompop;
  popid topop;
  double rate;
  struct migrate_struct* next;
} MigrateRate;

MigrateRate *migrations = NULL;
static double mig_lastrate = 0;

void 
migrate_add (popid from, popid to, double rate) 
{
  MigrateRate *newmigrate = malloc(sizeof (MigrateRate));
  assert(newmigrate != NULL);
  
  newmigrate->frompop = from;
  newmigrate->topop = to;
  newmigrate->rate = rate;
  newmigrate->next = migrations;
  
  migrations = newmigrate;
}

void 
migrate_delete (popid from, popid to) 
{
  MigrateRate *tempmigrate = migrations;
  MigrateRate *delmemigrate = NULL;
  int done = 0;
  
  if (tempmigrate == NULL) return;
  
  if (tempmigrate->frompop == from && 
      tempmigrate->topop == to) {
    migrations = migrations->next;
    done = 1;
  }
  
  while (tempmigrate != NULL &&
	 tempmigrate->next != NULL && !done) {
    if (tempmigrate->next->frompop == from && 
	tempmigrate->next->topop == to) {
      delmemigrate = tempmigrate->next;
      tempmigrate->next = tempmigrate->next->next;
      free(delmemigrate);
      done = 1;
    }
    tempmigrate = tempmigrate->next;
  }
}


double 
migrate_get_rate () 
{
  int numnodes, numnodes1;
  MigrateRate *tempmigrate = migrations;
  double rate = 0;
  
  if (migrations == NULL)
    mig_lastrate = 0;
  else {
    
    while (tempmigrate != NULL) {
      /* check if the pops exist. */
      
      numnodes = dg_get_num_nodes_in_pop_by_name (tempmigrate->frompop);
      /* change this into an existance check. */
      numnodes1 = dg_get_num_nodes_in_pop_by_name (tempmigrate->topop);
      /* numnodes = -1 when the pop does not exist. */
      if (numnodes < 0 || numnodes1 < 0) {
	migrate_delete (tempmigrate->frompop,
			tempmigrate->topop);
	return migrate_get_rate();
      }
      else {
	rate += numnodes * tempmigrate->rate;
	tempmigrate = tempmigrate->next;
      }
    }
    
    mig_lastrate = rate;
  }
  return mig_lastrate;
}


void 
migrate_execute (genid gen) 
{
  int numnodes;
  MigrateRate *tempmigrate = migrations;
  double rate = 0;
  double randcounter = random_double() * mig_lastrate;
  
  
  if (migrations == NULL)
    fprintf(stderr, "ERROR in migrate.\n");
  
  else {
    
    while (tempmigrate != NULL && rate < randcounter) {
      numnodes = dg_get_num_nodes_in_pop_by_name (tempmigrate->frompop);
      rate += numnodes * tempmigrate->rate;
      if (rate < randcounter)
	tempmigrate = tempmigrate->next;
      else
	dg_migrate_one_chrom (tempmigrate->frompop,
			      tempmigrate->topop, gen);
      
    }
  }
}
