/*
  The Broad Institute
  SOFTWARE COPYRIGHT NOTICE AGREEMENT
  This software and its documentation are copyright (2005) by the
  Broad Institute/Massachusetts Institute of Technology. All rights are
  reserved.

  This software is supplied without any warranty or guaranteed support
  whatsoever. Neither the Broad Institute nor MIT can be responsible for its
  use, misuse, or functionality.
*/
/* $Id: historical.c,v 1.13 2005/09/22 18:32:17 sfs Exp $ */

/*    historical.c
 *
 * 1. Processes historical events from datafile.
 * 2. Returns the time to the next historical event when called
 *    by simulator.c.
 * 3. Executes the historical event, by making calls to demography.c.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <math.h>
#include "defs.h"
#include "segment.h"
#include "node.h"
#include "nodelist.h"
#include "pop.h"
#include "demography.h"
#include "migrate.h"
#include "historical.h"
#include "sweep.h"

#define HE_LABEL_TOKENS "\""

#define HE_POP_SIZE 1
#define HE_POP_SIZE_STRING "change_size"

#define HE_POP_SIZE_EXP 2
#define HE_POP_SIZE_EXP_STRING "exp_change_size"

#define HE_BOTTLENECK 3
#define HE_BOTTLENECK_STRING "bottleneck"

#define HE_ADMIX 4
#define HE_ADMIX_STRING "admix"

#define HE_MIGRATION 6
#define HE_MIGRATION_STRING "migration_rate"

#define HE_SPLIT 7
#define HE_SPLIT_STRING "split"

#define HE_SWEEP 8
#define HE_SWEEP_STRING "sweep"

typedef struct historicalevent {
	genid gen;
	int type;
	int *popindex;
	double *params;
	char *label;
	struct historicalevent *next;
} hist_event;

#define EXP_INT 10
void historical_next_exp (hist_event * event);

void bottleneck_execute (int popindex, double coeff, int gen);

hist_event *currentevent = NULL;

int historical_numpops (int type);
int historical_numparams (int type);
void historical_new_event (int event_type, char * label, genid gen, int * newpops, double * newparams);
void historical_free (hist_event *);

int 
historical_numpops (int type ) 
{
	switch (type) {
	
	case HE_POP_SIZE:
		return 1;
		break;
	
	case HE_POP_SIZE_EXP:
		return 1;
		break;

	case HE_BOTTLENECK:
		return 1;
		break;
		
	case HE_ADMIX:
		return 2;
		break;

	case HE_MIGRATION:
		return 2;
		break;

	case HE_SPLIT:
		return 2;
		break;

	case HE_SWEEP:
		return 1;
		break;
	}

	return 0;
}



int 
historical_numparams (int type) 
{
	switch (type) {
	
	case HE_POP_SIZE:
		return 1;
		break;
	
	case HE_POP_SIZE_EXP:
	  return 3; /* end_gen, start_size, end_size */
		break;

	case HE_BOTTLENECK:
		return 1;
		break;
		

	case HE_ADMIX:
		return 1;
		break;

	case HE_MIGRATION:
		return 1;
		break;

	case HE_SPLIT:
		return 0;
		break;

	case HE_SWEEP:
		return 4;
		break;
	}
	 
	return 0;
}



int 
historical_get_type (char * typestr) 
{

	if (strcmp(typestr, HE_POP_SIZE_STRING) == 0) {
		return HE_POP_SIZE;
	}

	else if (strcmp(typestr, HE_POP_SIZE_EXP_STRING) == 0) {
		return HE_POP_SIZE_EXP;
	}

	else if (strcmp(typestr, HE_BOTTLENECK_STRING) == 0) {
		return HE_BOTTLENECK;
	}

	else if (strcmp(typestr, HE_ADMIX_STRING) == 0) {
		return HE_ADMIX;
	}

	else if (strcmp(typestr, HE_MIGRATION_STRING) == 0) {
		return HE_MIGRATION;
	}

	else if (strcmp(typestr, HE_SPLIT_STRING) == 0) {
		return HE_SPLIT;
	}

	else if (strcmp(typestr, HE_SWEEP_STRING) == 0) {
		return HE_SWEEP;
	}

	return -1;
}


genid 
historical_event_execute (genid historical_event_time) 
{
	hist_event *tempevent;
	genid gen = historical_event_time;

	if (currentevent == NULL) {
		fprintf(stderr, "sorry no historical event to execute");	
		return 0;
	}
	else {

		dg_log (HISTORICAL, currentevent->gen, currentevent->label);
		switch (currentevent->type) {
			
		case HE_POP_SIZE:
			dg_set_pop_size_by_name (currentevent->gen, 
						 currentevent->popindex[0], 
					 currentevent->params[0]);
			break;

		case HE_POP_SIZE_EXP:
			dg_set_pop_size_by_name (currentevent->gen,
						 currentevent->popindex[0], 
						 (int) (currentevent->params[1]+.5));

			historical_next_exp (currentevent);

			/* add a new pop size event */
			break;

		case HE_BOTTLENECK:
			bottleneck_execute (currentevent->popindex[0], 
					    currentevent->params[0],
					    currentevent->gen); 
			/*			gen += 1; */
			break;

		case HE_ADMIX:
		  /* 27-sep-04.  Modified to simply shift the required # chroms to another pop. */

		  /* join two pops forward = split pops backwards */
		  /* change pop size to what is specified. */
		  /*		  		  dg_set_pop_size_by_name (currentevent->gen,
					   currentevent->popindex[1],
					   currentevent->params[0]);  */
		  
		  dg_move_nodes_by_name (currentevent->popindex[0],
					 currentevent->popindex[1],
					 currentevent->params[0],
					 currentevent->gen);
		  break;
		  
		  
		case HE_MIGRATION:
		  migrate_delete (currentevent->popindex[1],
				  currentevent->popindex[0]);
		  migrate_add (currentevent->popindex[1],
			       currentevent->popindex[0],
			       currentevent->params[0]);
		  break;
		  
		case HE_SPLIT:
		  /* split two pops forward = join pops backwards */
		  dg_move_nodes_by_name (currentevent->popindex[1],
					 currentevent->popindex[0],
					 1,
					 currentevent->gen);
		  dg_end_pop_by_name (currentevent->popindex[1]);
		  break;

		case HE_SWEEP:
		  gen = sweep_execute (currentevent->popindex[0], 
				       currentevent->params[0],
				       currentevent->gen, currentevent->params[1],
				       (int) (currentevent->params[2]), currentevent->params[3]); 
		  break;
		}

		tempevent = currentevent;
		currentevent = currentevent->next;
		historical_free(tempevent);
		
		return gen;
	}

		
}

void historical_free (hist_event *event) {
	free(event->label);
	event->label = NULL;
	free(event->popindex);
	event->popindex = NULL;
	if (event->params != NULL) {
		free(event->params);
		event->params = NULL;
	}

	
	free(event);
	event = NULL;
} 

genid historical_get_next (genid gen) {
	if (currentevent == NULL)
		return -1;
	else return currentevent->gen - gen;
}

void historical_process_pop_event (char* buffer) {

	double *newparams;
	int *newpops;
	int i = 0;
	int event_type;
	char *labelstr, *newlabel;
	char *typestr;
	genid gen;
	double k;

	typestr = strtok (buffer, " ");

	if (DEBUG)
		printf("type: %s\n", typestr);

	event_type = historical_get_type (typestr);

	/* extract event label (inside quotes) */
	labelstr = strtok (NULL, HE_LABEL_TOKENS);
	
	newlabel = malloc (sizeof (char) * (strlen(labelstr) + 1));

	strcpy (newlabel, labelstr);
	if (DEBUG)
		printf("event label: %s\n", newlabel);

	/* get pops */
	newpops = malloc (sizeof (popid) * historical_numpops (event_type));
	for (i = 0; i < historical_numpops (event_type); i++) {
		newpops[i] = atoi(strtok (NULL, " "));
		
	}

	/* extract gen */
	gen = atof(strtok (NULL, " "));
	/* get numerical params */
	if (historical_numparams (event_type) > 0) {
		newparams = malloc (sizeof (double) * historical_numparams (event_type)); 

		for (i = 0; i < historical_numparams (event_type); i++) {
			newparams[i] = atof(strtok (NULL, " "));
		}
	}
	else newparams = NULL;

	if (event_type == HE_POP_SIZE_EXP) {
	  /*	  fprintf(stderr, "tstart: %f  tend: %f  start size: %f  end size: %f\n", gen, newparams[0], 
		  newparams[2], newparams[1]); */
		k = log(newparams[2]/newparams[1]) / (newparams[0] - gen);
		newparams[2] = k;
		/* we no longer need to know what the ending pop size is, 
		   since theoretically that's where we'll end up */
	}

	/* create new historical event */
	historical_new_event (event_type, newlabel, gen, newpops, newparams);
}

hist_event * historical_add (hist_event *newevent, hist_event *eventlist) {
	if (eventlist == NULL) {
		newevent->next = eventlist;
		return newevent;
	}

	else if (newevent->gen < eventlist->gen) {
		newevent->next = eventlist;
		return newevent;
	}
	/*	else if (newevent->gen == eventlist->gen) {
	  fprintf(stderr, "Two historical events (%s & %s) at identical time %f.  Exiting\n", 
		  newevent->label, eventlist->label, eventlist->gen);
	  fprintf(stderr, "Note that event times do not have to be integers.\n");
	  exit(0);
	  }*/
	else {
		eventlist->next = historical_add (newevent, eventlist->next);
		return eventlist;
	}
}

void 
historical_new_event (int event_type, char *label, genid gen, 
			   int *newpops, double *newparams) 
{
	hist_event *newevent;

	newevent = malloc (sizeof (*newevent));

	newevent->gen = gen;
	newevent->type = event_type;
	newevent->popindex = newpops;

	newevent->params = newparams;

	newevent->label = label;
	newevent->next = NULL;

	currentevent = historical_add (newevent, currentevent);
}


/* goes with HE_POP_SIZE_EXP */
void 
historical_next_exp (hist_event *event) 
{
	genid gen;
	char * label;
	int * newpops;
	double * newparams;

	if (event->gen == event->params[0]) return;
	/* we're done! */
	assert (event->gen < event->params[0]);

	label = malloc (sizeof (char) * (strlen(event->label) + 1));
	newpops = malloc (sizeof (int) * 1);
	newparams = malloc (sizeof (double) * 3);

	strcpy (label, event->label);
	newpops[0] = event->popindex[0];

	gen = event->gen + EXP_INT;
	if (gen > event->params[0])
	  { gen = event->params[0]; }
	
	newparams[0] = event->params[0];
	newparams[1] = event->params[1] * 
	  exp (event->params[2] * (gen - event->gen));
	newparams[2] = event->params[2]; /* expansion rate */
	/*  fprintf(stderr, "next time: %f this time: %f new size: %f  k: %f\n", gen, event->gen, newparams[1], newparams[2]);*/
	
	historical_new_event (event->type, label, gen, newpops, newparams);
}
