/*
  The Broad Institute
  SOFTWARE COPYRIGHT NOTICE AGREEMENT
  This software and its documentation are copyright (2005) by the
  Broad Institute/Massachusetts Institute of Technology. All rights are
  reserved.

  This software is supplied without any warranty or guaranteed support
  whatsoever. Neither the Broad Institute nor MIT can be responsible for its
  use, misuse, or functionality.
*/
/* $Id: file.c,v 1.10 2005/09/22 18:32:17 sfs Exp $ */
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include "defs.h"
#include "segment.h"
#include "node.h"
#include "nodelist.h"
#include "pop.h"
#include "demography.h"
#include "mutlist.h"
#include "haplos.h"
#include "simulator.h"
#include "geneconversion.h"
#include "historical.h"
#include "recomb.h"
#include "../cosi_rand/random.h"
#include "output.h"

#define BUF_MAX 100

static int seeded = 0;
int popsize;
int sampsize;
int length;
double mu;
double recombrate;
char poplabel[30];
long rseed;

int file_read(char* filename, FILE*);
int file_get_data (FILE * fileptr, FILE *);
int file_proc_buff(char * var, char* buffer, FILE*);
int file_killwhitespace(FILE * fileptr);
void file_proc_recombfile (char*  filename);
void file_exit(char* , char*);
void file_error_nonfatal(char* , char*);

int 
file_read(char* filename, FILE *segfp) 
{
	FILE *infileptr = fopen (filename, "r");

	if (infileptr == NULL)
		file_exit("file_read", "cannot open file for reading");

	if (segfp != NULL) {fprintf(segfp, "paramfile: %s\n", filename);}
	file_get_data (infileptr, segfp);
	fclose(infileptr);	
	return length;
}

/*********************************************************/

int 
file_proc_buff(char *var, char* buffer, FILE* segfp) 
{

	int     popname, 
		intarg;
	char   *tempstr, 
	       *str1;
		
	/*	printf("%s\n", var); */
	if (strcmp(var,"length") == 0) {
		length = atoi(buffer);
		sim_setlength(length);
	}
	else if (strcmp(var,"recomb_file") == 0) {
		if (buffer[strlen(buffer)-1] == '\n')
			buffer[strlen(buffer)-1] = '\0';
		file_proc_recombfile(buffer);
	}
	else if (strcmp(var, "mutation_rate") == 0) {
		mu = atof(buffer);
	}
	else if (strcmp(var, "infinite_sites") == 0) {
	  if (buffer[strlen(buffer)-1] == '\n')
	    buffer[strlen(buffer)-1] = '\0';
	  if (strcmp(buffer, "yes") == 0 || strcmp(buffer, "Yes") == 0 || strcmp(buffer, "YES") == 0) {
	    out_set_infinite_sites(1);
	  }
	}
	else if (strcmp(var, "gene_conversion_rate") == 0) {
		gc_set_rate(atof(buffer));
	}
	else if (strcmp(var, "number_mutation_sites") == 0) {
	  sim_set_num_muts(atoi(buffer));
	}
	else if (strcmp(var, "pop_label") == 0) {
		if (buffer[strlen(buffer)-1] == '\n')
			buffer[strlen(buffer)-1] = '\0';
		strcpy(poplabel, buffer);
		if (FILE_DEBUG)
			printf("pop label: %s\n", poplabel);
	}
	else if (strcmp(var, "pop_size") == 0) {
		popname = atoi(strtok (buffer, " "));
		intarg = atoi(strtok (NULL, " " ));
		if (FILE_DEBUG)
			printf("popsize: %d\n", intarg);
		/* 
		 * Throw a fatal error if pop [popname] does not exist.
		 */		
		if (! dg_set_pop_size_by_name (0, popname, intarg))
			file_exit("file_proc_buff", 
				   "parameter file - pop specified does not exist.");
	}
	else if (strcmp(var, "sample_size") == 0) {		
		popname = atoi(strtok (buffer, " "));
		intarg = atoi(strtok (NULL, " " ));
		if (FILE_DEBUG)
			printf("sampsize: %d\n", intarg);
		dg_populate_by_name (popname, intarg, 0);
		if (segfp != NULL) {fprintf(segfp, "A %d %d\n", popname, intarg);}
	}
	else if (strcmp(var, "pop_define") == 0) {
		popname = atoi(strtok (buffer, " "));
		tempstr = strtok (NULL, " ");
		str1 = malloc (strlen(tempstr) + 1);
		strncpy(str1, tempstr,strlen(tempstr)+1);
		dg_create_pop (popname, str1, 0);
	}
	else if (strcmp(var, "pop_event") == 0) {
		historical_process_pop_event (buffer);
	}
	else if (strcmp(var, "random_seed") == 0) {
	  if (atof(buffer) > 0) {
	    rseed = -1 * atof(buffer);
	    set_rng_seed(rseed);
	    fprintf(stdout, "coalescent seed set in param file: %ld\n", -1 * rseed);
	    seeded = 1;
	  }
	}
	else {
		file_error_nonfatal("file_proc_buffer", 
				    "unknown variable");
		if (FILE_DEBUG)
			printf("%s is not a valid variable\n", var);
	}
	
	return 1;
}

void 
file_proc_recombfile (char* filename) 
{
	FILE *recombfp = fopen(filename, "r");
	int start;
	double rate;

	if (recombfp != NULL) {
		while (fscanf(recombfp, "%d %lf", &start, &rate) == 2) {
			recomb_add(start, rate);
		}
		fclose(recombfp);
		recomb_calc_r();
	}
	else {
		file_exit("file_proc_recombfile","can't read file for recomb");
	}
}

int 
file_get_data (FILE *fileptr, FILE *segfp) 
{
	char c,
		buffer[BUF_MAX],
		var[50];

	c = getc(fileptr);
	while (c != EOF) {
		switch (c) {
		case '#':
			c = getc(fileptr);
			while (c != '\n' && c != EOF)
				c = getc(fileptr);
			break;
		case '\n':
			c = getc(fileptr);
			break;
		case ' ':
			c = getc(fileptr);
			break;
		default:
			ungetc(c, fileptr);
			fscanf(fileptr, "%s", var);
			file_killwhitespace(fileptr);
			fgets(buffer, BUF_MAX, fileptr);
			file_proc_buff(var, buffer, segfp);
			c = getc(fileptr);
			break;
		}
	}
	sim_settheta(mu * length);
	if (seeded != 1) {
	  printf("coalescent seed: %ld\n", -1 * seed_rng());
	}
	/*
	  printf("theta set to: %f\n", mu*length);
	  printf("length: %d\n", length);
	  printf("mu: %.9f\n", mu);
	*/


	if (segfp != NULL) {fprintf(segfp, "params: length %d mu %.9f\n", length, mu);}
	if (segfp != NULL) {fprintf(segfp, "L %d\n", length);}

	return 1;
}

int 
file_killwhitespace(FILE * fileptr) 
{
	char c;
	int i = 0;
	c = getc(fileptr);
	while (isspace((int) c)) {
		c = getc(fileptr);
		i++;
	}
	ungetc(c, fileptr);
	return i;
}

/*
 * FILE_EXIT
 */
void 
file_exit(char* funct_name, char* error_string) 
{
	fprintf(stderr, "file.c | %s: %s\n",
		funct_name, error_string);
	exit(EXIT_FAILURE);
}

void 
file_error_nonfatal(char* funct_name, char* error_string) 
{
	fprintf(stderr, "file.c | %s: %s\n",
		funct_name, error_string);
}
