/*
  The Broad Institute
  SOFTWARE COPYRIGHT NOTICE AGREEMENT
  This software and its documentation are copyright (2005) by the
  Broad Institute/Massachusetts Institute of Technology. All rights are
  reserved.

  This software is supplied without any warranty or guaranteed support
  whatsoever. Neither the Broad Institute nor MIT can be responsible for its
  use, misuse, or functionality.
*/
/* $Id: demography.c,v 1.13 2006/01/18 19:26:25 sfs Exp $ */

#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <assert.h>
#include "defs.h"
#include "segment.h"
#include "node.h"
#include "nodelist.h"
#include "pop.h"
#include "mutlist.h"
#include "haplos.h"
#include "mutate.h"
#include "../cosi_rand/random.h"
#include "../cosi_rand/ranbinom.h"
#include "demography.h"

static int LOGGING = 1;
FILE* logfp;

/* data structure definitions */

typedef struct poplist 
{
	Pop *pop;
	struct poplist *next;
} poplist;

struct demography 
{
  struct sitelist *recombsites;
  struct poplist *pops;
  Pop* completepop;
  int numpops;
  rsitenum numsites;
  int totmembers;
};

struct demography demography_data;

struct sitelist * dg_recombsites(void) {
  return demography_data.recombsites;
}


/* ERROR HANDLING */
void dg_exit (char *funct_name, char *error_string);
void dg_error_nonfatal (char *funct_name, char *error_string);

/* POP FUNCTIONS */
Pop * dg_get_pop_from_list(int index, poplist* poplistptr);
void dg_add_pop (Pop *, genid gen, struct poplist **);
Pop * dg_get_pop_by_name_int (int, struct poplist *);
struct poplist * dg_delete_pop_by_name (popid popname, 
					struct poplist * temppoplist) ;
int dg_add_to_completepop (Node * nodeptr, Pop *, 
			   double begin, double end);

struct sitelist * dg_add_recomb_site (double site, 
				      struct sitelist *sitelistptr);

/* MUTATION */
genid dg_calc_time_in_branch (double /* begin */, genid /* parenttime */, 
			      Node* /* nodeptr */);

/* DG_INITIALIZE 
 * Must be called only once at the beginning of the program.
 */
void 
dg_initialize() 
{
	/* Initialize a bunch of values. 
	 */
	demography_data.pops = NULL;
	demography_data.numpops = 0;
	demography_data.totmembers = 0;

	/* This is the population for nodes which have been completed. */
	demography_data.completepop = pop_new (-1, 0, "complete_nodes");
}

/* DG_INIT_RECOMB
 * Must be called only once, after sample sizes set.
 */
void 
dg_init_recomb() 
{
	demography_data.recombsites = dg_add_recomb_site(0.0, NULL);
	demography_data.recombsites = 
		dg_add_recomb_site(1.0, demography_data.recombsites);
	demography_data.numsites = 1;
}

/* POP FUNCTIONS 
 * "index" refers to the location in the population list.
 * "name" refers to the numerical name of the population, as
 *        defined in the param file.
 * 
 * How to create a population:
 * 1. call dg_create_pop to create an empty population
 * 2. call dg_set_pop.. to set the population size
 * 3. (opt) call dg_populate_by_name to fill in the nodes
 */
void
dg_create_pop (popid popname, char *label, genid gen) {
	Pop *newpop, *temppop;
	
	/* check to make sure the popname is not taken and
	 is non negative
	 if it is, throw a fatal exception.  */
	if (popname < 0) {
		dg_exit("dg_create_pop","popname must not be negative");
	}
	temppop = dg_get_pop_by_name_int(popname, demography_data.pops);

	if (temppop != NULL) {
		dg_exit("dg_create_pop","duplicate popname used.");
	}

	newpop = pop_new (popname, 0, label);

	dg_add_pop (newpop, gen, &demography_data.pops);

}

void
dg_populate_by_name (popid popname, int members, genid gen) {
	Pop *popptr = dg_get_pop_by_name_int(popname, demography_data.pops);
	Node *tempnode;
	int i;	
	
	for (i = 0; i < members; i++) {
		tempnode = make_new_node (0,1,gen,popname);
		pop_add_node (popptr, tempnode);
		dg_log (ADD_NODE, gen, tempnode, popptr);
	}	
	demography_data.totmembers += members;
}

int 
dg_get_num_pops (void) 
{
	return demography_data.numpops;
}

char * 
dg_get_pop_label_by_name (popid popname) 
{
	return dg_get_pop_by_name_int(popname, demography_data.pops) -> label;
}

/* returns -1 if something goes wrong.  */
int 
dg_get_pop_name_by_label (char *label) 
{
	struct poplist *temppop = demography_data.pops;

	while (temppop != NULL) {
		if (strcmp(temppop->pop->label, label) == 0)
			return temppop->pop->name;
		temppop = temppop->next;
	}

	return -1;
}

/* POP_SIZE FUNCTIONS */
int 
dg_get_pop_size_by_index (int popindex) 
{
	Pop *popptr =  dg_get_pop_by_index(popindex);
	return popptr->popsize;
}

/*
 * Returns zero if the population specified does not exist.
 */
int 
dg_set_pop_size_by_name (genid gen, popid popname, int newsize) 
{
	Pop *popptr =  dg_get_pop_by_name_int(popname, demography_data.pops);
	if (popptr == NULL) {
		dg_error_nonfatal("dg_set_pop_size_by_name", 
				  "pop does not exist");
		return 0;
	}
	pop_set_size(popptr, newsize);
	dg_log(CHANGE_SIZE, gen, popptr);
	return 1;
}

/* ENDING POPS */

void 
dg_end_pop_by_name (popid popname) 
{
	demography_data.pops = dg_delete_pop_by_name (popname, 
						      demography_data.pops);
}

/* COALESCE */

void
dg_coalesce_by_index (int popindex, genid gen) 
{
  assert(popindex >= 0);
  Pop *popptr = dg_get_pop_by_index (popindex);
  dg_coalesce_by_name (popptr->name, gen);
}

void 
dg_coalesce_by_name(popid popname, genid gen) 
{
	Pop *popptr;
	int node1index, node2index, nodesatloc, contains;
	Node *node1, *node2, *newnode;
	struct sitelist * sitetemp = demography_data.recombsites;
	double loc;
	Segment *tseg;
  
	/* 
	 * 1. Choose two unique nodes.
	 * 2. Coalesce them, creating newnode.
	 * 2a. Update the tally of nodes associated with segments
	 * 3. Remove old nodes from population.
	 * 4. Add new node to population.
	 * 5. Log it.
	 */ 

	popptr = dg_get_pop_by_name_int (popname, demography_data.pops);

	/* STEP 1 */
	node1index = (int) (random_double() * popptr->members.nummembers);
	node2index = (int) (random_double() * (popptr->members.nummembers - 1));

	if (node2index >= node1index) node2index++;

	node1 = pop_get_node (node1index, &(popptr->members));
	node2 = pop_get_node (node2index, &(popptr->members));

	/* STEP 2 */
	newnode = node_coalesce (node1, node2, gen);

	/* STEP 2a */
	while (sitetemp->next != NULL) {
	  if (sitetemp->nnode >= 2) {
	    loc = (sitetemp->site + sitetemp->next->site) / 2;
	    nodesatloc = 0;
	    tseg = node1->segs;
	    contains = 0;
	    while (tseg != NULL) {
	      if (loc >= tseg->begin) {
		if (loc <= tseg->end) {
		  contains = 1;
		  break;
		}
		tseg = tseg->next;
	      }
	      else { 
		contains = 0;
		break;
	      }
	    }
	    if (contains) {
	      nodesatloc++;
	    }
	    tseg = node2->segs;
	    contains = 0;
	    while (tseg != NULL) {
	      if (loc >= tseg->begin) {
		if (loc <= tseg->end) {
		  contains = 1;
		  break;
		}
		tseg = tseg->next;
	      }
	      else { 
		contains = 0;
		break;
	      }
	    }
	    if (contains) {
	      nodesatloc++;
	    }
	    if (nodesatloc > 1) {
	      sitetemp->nnode--;
	    }
	  }
	  sitetemp = sitetemp->next;
	}

	/* STEP 3 */
	pop_remove_node (popptr, node1);
	pop_remove_node (popptr, node2);
	
	/* STEP 4 */
	pop_add_node (popptr, newnode);

	/* STEP 5 */
	dg_log (COALESCE, gen, node1, node2, newnode, popptr);
}

/* RECOMBINE */

Node**
dg_recombine_by_index (int popindex, genid gen, double loc)
{
	Node    *newnode1 = NULL, 
		*newnode2 = NULL, 
		*node;
	Node    **returnNodes = NULL;
	Pop     *popptr;
	int     nodeindex, 
		nr;

	/*
	 * 1. Pick a random node to recombine.
	 * 2. Execute recombination.
	 * 3. If recombination produces two nodes (i.e.
	 *    recombination occurs in one of these locations:
	 *        a. in the middle of an "active segment", or
	 *        b. between two "active segments",
	 *    then do following steps, otherwise exit.
	 * 4. Remove old node.
	 * 5. Add two new nodes.
	 * 6. Add new recombination site.
	 * 7. Log it.
	 */

	popptr = dg_get_pop_by_index (popindex);
	
	/* STEP 1 */
	nodeindex = (int) (random_double() * popptr->members.nummembers);
	node = pop_get_node (nodeindex, &(popptr->members));

	/* STEP 2 */
	nr = node_recombine (node, &newnode1, &newnode2, gen, loc);

	/* STEP 3 */
	if (nr == 2) {
	        /* STEP 4 */
		pop_remove_node (popptr, node);

		/* STEP 5 */
		pop_add_node (popptr, newnode1);
		pop_add_node (popptr, newnode2);
		
		/* STEP 6 */
		demography_data.recombsites = 
			dg_add_recomb_site (loc, demography_data.recombsites);

		/* STEP 7 */
		dg_log (RECOMBINE, gen, node, newnode1, newnode2, popptr, loc);

		returnNodes = malloc(3*sizeof(Node**));
		returnNodes[0] = node;
		returnNodes[1] = newnode1;
		returnNodes[2] = newnode2;
		return returnNodes;
	}
	else {
	  return NULL;
	}

}

/* GENE CONVERSION */

Node**
dg_gc_by_index (int popindex, genid gen, double loc, double locend)
{
	Node    *newnode1 = NULL, 
		*newnode2 = NULL, 
		*node;
	Node    **returnNodes = NULL;
	Pop     *popptr;
	int     nodeindex, 
		nr;

	/*
	 * 1. Pick a random node to geneconvert.
	 * 2. Execute recombination.
	 * 3. If recombination produces two nodes (i.e.
	 *    the location of gene conversion overlaps an
	 *    active region)
	 *    then do following steps, otherwise exit.
	 * 4. Remove old node.
	 * 5. Add two new nodes.
	 * 6. Add two new recombination sites.
	 * 7. Log it.
	 */

	popptr = dg_get_pop_by_index (popindex);
	
	/* STEP 1 */
	nodeindex = (int) (random_double() * popptr->members.nummembers);
	node = pop_get_node (nodeindex, &(popptr->members));

	/* STEP 2 */
	nr = node_gc (node, &newnode1, &newnode2, gen, loc, locend);

	/* STEP 3 */
	if (nr == 2) {
		/* STEP 4 */
		pop_remove_node (popptr, node);

		/* STEP 5 */
		pop_add_node (popptr, newnode1);
		pop_add_node (popptr, newnode2);
		
		/* STEP 6 */
		if (seg_contains(node->segs, loc))

			demography_data.recombsites = 
				dg_add_recomb_site (loc, demography_data.recombsites);

		if (seg_contains(node->segs, locend))
			demography_data.recombsites = 
				dg_add_recomb_site (locend, demography_data.recombsites);

		/* STEP 7 */
		dg_log (GENE_CONVERSION, gen, node, newnode1, newnode2, popptr, loc, locend);
		returnNodes = malloc(3*sizeof(Node**));
		returnNodes[0] = node;
		returnNodes[1] = newnode1;
		returnNodes[2] = newnode2;
		return returnNodes;
	}
	else {
	  return NULL;
	}
	
}

/* MIGRATE */

void 
dg_migrate_one_chrom (popid frompop, popid topop, genid gen) 
{
	Pop     *from_popptr, 
		*to_popptr;
	Node *tempnode;
	int node_index;

	from_popptr = dg_get_pop_by_name_int (frompop, demography_data.pops);
	to_popptr = dg_get_pop_by_name_int (topop, demography_data.pops);
	
	node_index = (int) (random_double() * pop_get_num_nodes(from_popptr));
	tempnode = pop_get_node (node_index, &(from_popptr->members));
	pop_remove_node (from_popptr, tempnode);
	pop_add_node (to_popptr, tempnode);
	dg_log(MOVE, gen, tempnode, from_popptr, to_popptr);
	
}


/* MOVING NODES */
/* dg_move_nodes_by_name is called from functions that implement 
 * admixing and spliting, to move a bunch of nodes from one 
 * population to another. The number moved is binomially distributed.
 */
void 
dg_move_nodes_by_name (int frompop, int topop, double members, genid gen) 
{
	Pop     *from_popptr, 
		*to_popptr;
	Node *tempnode;
	int num_to_move;
	int node_index, i;

	from_popptr = dg_get_pop_by_name_int (frompop, demography_data.pops);
	to_popptr = dg_get_pop_by_name_int (topop, demography_data.pops);
	
	num_to_move = ranbinom(pop_get_num_nodes(from_popptr), members);

	for (i = 0; i < num_to_move; i++) {
		node_index = (int) (random_double() 
				    * pop_get_num_nodes(from_popptr));
		tempnode = pop_get_node (node_index, &(from_popptr->members));
		pop_remove_node (from_popptr, tempnode);
		pop_add_node (to_popptr, tempnode);
		dg_log(MOVE, gen, tempnode, from_popptr, to_popptr);
	}
}

/* NODE FUNCTIONS */

/* GET_NUM_NODES */
int 
dg_get_num_nodes (void) 
{
	int     total = 0,
		i;

	for (i = 0; i < demography_data.numpops; i++)
		total = dg_get_num_nodes_in_pop_by_index (i);
	return total;
}

int 
dg_get_num_nodes_in_pop_by_index (int popindex)
{
	Pop * popptr = dg_get_pop_by_index ( popindex);
	if (popptr == NULL) return -1;
	return popptr->members.nummembers;
}

int 
dg_get_num_nodes_in_pop_by_name (popid popname){
	Pop * popptr = dg_get_pop_by_name_int ( popname, demography_data.pops);
	if (popptr == NULL) return -1;
	return popptr->members.nummembers;
}

/* DG_DONE_COALESCENT */
/*
 *  checks if the coalescent is done. this needs to take every recomb
 *  site and see if there is more than one node at the top that contains
 *  it. there should also only be one population at this point. Returning
 *  1 means we're done, 0 means we're not done.
 */
int 
dg_done_coalescent () 
{
  struct sitelist * sitetemp = demography_data.recombsites;
  int     nonemptypopcount = 0, i, contains;
  Pop     *popptr=NULL, *temppopptr;
  Node    *nodeptr, *tempnodeptr=NULL;
  double loc;
  Segment *tseg;

	/*
	 * so i don't want to delete the pops right away, but i don't
	 * want to count empty pops either. so we count non-empty pops
	 * instead of deleting them. If there is more than one non-empty
	 * pop, we know we're not done.
	*/

 	for (i = demography_data.numpops - 1; i >= 0; i--) {
		temppopptr = dg_get_pop_by_index(i);
		if ( pop_get_num_nodes( temppopptr) > 0) {  
			nonemptypopcount++;
			popptr = temppopptr;
		}
	}

	if (nonemptypopcount > 1) return 0;

	/* 
	 * Now check each member of the last population and move
	 * out all the regions that have coalesced. 
	 * The regions moved out go into a "completepop" where they
	 * do not participate in further simulation and await mutation.
	 * If the entire chromosome is coalesced, we're done.
	 */

	if (popptr->members.nummembers > 1) {      
	  while (sitetemp->next != NULL) {
	    if (sitetemp->nnode == 1) {
	      loc = (sitetemp->site + sitetemp->next->site) / 2;
	      for (i = 0; i < popptr->members.nummembers; i++) {
		/* Unoptimized:*/
		/* nodeptr = pop_get_node (i, &(popptr->members)); */
		/* Optimized, but fragile: */
		nodeptr = popptr->members.nodes[popptr->members.nummembers - i - 1]; 
		/* Start hardwired optimization (from seg_contains(), segment.c) */
		tseg = nodeptr->segs;
		contains = 0;
		while (tseg != NULL) {
		  if (loc >= tseg->begin) {
		    if (loc <= tseg->end) {
		      contains = 1;
		      tempnodeptr = nodeptr;
		      break;
		    }
		    tseg = tseg->next;
		  }
		  else { 
		    contains = 0;
		    break;
		  }
		}
		/* end optimization */
		assert(contains = 1);
	      }
	      dg_add_to_completepop (tempnodeptr, 
				     popptr, 
				     sitetemp->site, 
				     sitetemp->next->site);
	      sitetemp->nnode = 0;
	    }
	    sitetemp = sitetemp->next;
	  }
	}

	if (popptr->members.nummembers < 2) {
		tempnodeptr = popptr->members.nodes[0];

		pop_add_node(demography_data.completepop,
			     tempnodeptr);
		pop_remove_node(popptr, tempnodeptr);

		if (DEMOG_DEBUG) {
			fprintf(stdout, 
				"the following %d nodes are left:\n", 
				demography_data.completepop->members.nummembers);
			for (i = 0; 
			     i < demography_data.completepop->members.nummembers; i++)
				print_node(pop_get_node 
					   (i, 
					    &(demography_data.completepop->members)));
		}  
		return 1;
	}

	return 0;
}


/* LOGGING */
/* The dg_log function can take a variable number of arguments,
 * depending on what we are logging. The only external call of this
 * function occurs in historical.c, where it is needed to log the
 * historical events that occur. All other calls are from within
 * demography.c.
 */
void 
dg_log (int type, genid gen, ...) {
	FILE   *outputptr = logfp;
	Pop     *popptr, 
		*popptr2;
	Node    *nodeptr1, 
		*nodeptr2, 
		*nodeptr3;
	double  double1;
	double  loc, loc2;
	char*   string1;
	va_list ap; /* points to unnamed args */

	if (LOGGING) {

		va_start(ap, gen);

		switch (type) {
		case ADD_NODE: 
			/* ADD_NODE newnode pop */
			nodeptr1 = va_arg(ap, Node *);
			popptr = va_arg(ap, Pop *);
			if (outputptr != NULL) {fprintf(outputptr, 
							"%f\tADD\tnode: %d pop: %d\n", 
							gen, nodeptr1->name, popptr->name);}
			break;
      
		case CHANGE_SIZE: 
			/* CHANGE_SIZE pop */
			popptr = va_arg(ap, Pop*);
			if (outputptr != NULL) {fprintf(outputptr, 
							"%f\tchange_size\tpop: %d size: %d\n", 
							gen, popptr->name, popptr->popsize);}
			break;

		case COALESCE:
			/* COALESCE oldnode1 oldnode2 newnode pop */
			nodeptr1 = va_arg(ap, Node *);
			nodeptr2 = va_arg(ap, Node *);
			nodeptr3 = va_arg(ap, Node *);
			popptr = va_arg(ap, Pop *);
			if (outputptr != NULL) {fprintf(outputptr, "%f\tC\t%d %d -> %d pop: %d\n", 
							gen, nodeptr1->name, nodeptr2->name, 
							nodeptr3->name, popptr->name);}
			break;
      
		case CREATE_POP:
			/* CREATE_POP pop */
			popptr = va_arg(ap, Pop*);
			if (outputptr != NULL) {fprintf(outputptr, 
							"%f\tcreate_pop\tpop: %d size: %d\n", 
							gen, popptr->name, popptr->popsize);}
			break;

		case DONE:
			/* DONE oldnode donenode [newnode|NULL] */
			nodeptr1 = va_arg(ap, Node *);
			nodeptr2 = va_arg(ap, Node *);
			nodeptr3 = va_arg(ap, Node *);

			if (nodeptr3 != NULL) {
			  if (outputptr != NULL) {fprintf(outputptr, 
							  "%f\tD\t%d -> %d | %d %f %f\n",
							  gen, nodeptr1->name, 
							  nodeptr3->name, 
							  nodeptr2->name, 
							  nodeptr2->segs->begin, 
							  nodeptr2->segs->end);}
			}
			else {
			  if (outputptr != NULL) {fprintf(outputptr, "%f\tD\t%d\n", gen,
							  nodeptr1->name);}
			}
			break;

		case GENE_CONVERSION:
			/* RECOMBINE oldnode1 oldnode2 newnode pop */
			nodeptr1 = va_arg(ap, Node*);
			nodeptr2 = va_arg(ap, Node *);
			nodeptr3 = va_arg(ap, Node *);
			popptr = va_arg(ap, Pop*);
			loc = va_arg(ap, double);
			loc2 = va_arg(ap, double);
			
			if (nodeptr3 != NULL) {
			  if (outputptr != NULL) {fprintf(outputptr, 
							  "%f\tG\t%d -> %d %d %d %f %f\n", 
							  gen, nodeptr1->name, 
							  nodeptr2->name, 
							  nodeptr3->name,
							  popptr->name, loc, loc2);}
			}
			else {
			  if (outputptr != NULL) {fprintf(outputptr, 
							  "%f\tG\t%d -> %d  - %d %f %f\n", 
							  gen, nodeptr1->name, 
							  nodeptr2->name, 
							  popptr->name, loc, loc2);}
			}
			break;
			


		case HISTORICAL:
			/* HISTORICAL string_description */
			string1 = va_arg(ap, char*);
			if (outputptr != NULL) {fprintf(outputptr, "%f\tH\t%s\n",
							gen, string1);}
			break;

		case MIG_RATE:
			/* MIG_RATE from-pop to-pop new_rate */
			/* note that "from" and "to" are in real time.
			 * chromosomes will move in the opposite direction */
			popptr = va_arg(ap, Pop *);
			popptr2 = va_arg(ap, Pop *);
			double1 = va_arg(ap, double);
			if (outputptr != NULL) {fprintf(outputptr, 
							"%f\tmig_rate\t%d %d %f\n", 
							gen, popptr->name, 
							popptr2->name, double1);}
			break;

		case MOVE:
			/* MOVE node from-pop to-pop */
			nodeptr1 = va_arg(ap, Node *);
			popptr = va_arg(ap, Pop *);
			popptr2 = va_arg(ap, Pop *);			
			if (outputptr != NULL) {fprintf(outputptr, "%f\tM\t%d %d %d\n",
							gen, nodeptr1->name, popptr->name, 
							popptr2->name);}
			break;

		case RECOMBINE:
			/* RECOMBINE oldnode1 oldnode2 newnode pop */
			nodeptr1 = va_arg(ap, Node*);
			nodeptr2 = va_arg(ap, Node *);
			nodeptr3 = va_arg(ap, Node *);
			popptr = va_arg(ap, Pop*);
			loc = va_arg(ap, double);

			if (nodeptr3 != NULL) {
			  if (outputptr != NULL) {fprintf(outputptr, 
							  "%f\tR\t%d -> %d %d %d %f\n", 
							  gen, nodeptr1->name, 
							  nodeptr2->name, 
							  nodeptr3->name,
							  popptr->name, loc);}
			}
			else {
			  if (outputptr != NULL) {fprintf(outputptr, 
							  "%f\tR\t%d -> %d  - %d %f\n", 
							  gen, nodeptr1->name, 
							  nodeptr2->name, 
							  popptr->name, loc);}
			}
			break;
		       
		}
	}	
}

void 
dg_set_logfile (FILE *fp) 
{
	logfp = fp;
}

void 
dg_logging_on () 
{
	LOGGING = 1;
}

void 
dg_logging_off () 
{
	LOGGING = 0;
}



/***********************************************************/
/*  FUNCTIONS FOR MUTATION */
/*  see mutate.c */

rsitenum 
dg_get_num_regs (void) 
{
	return demography_data.numsites;
}

/*  returns the length of the region indicated by rindex.
 *  does the exterior program need to know the position of the region?
 *  perhaps this can be used later to modify mutation rates on different
 *  parts of the chromosome.
 */
double 
dg_get_reg_length (rsitenum rindex1) 
{
	int i = 0;
	struct sitelist *tempsites = demography_data.recombsites;

	if (rindex1 > demography_data.numsites) {
		printf("rindex too long\n");
		return 0;
	}

	while (i < rindex1 && tempsites != NULL) {
		tempsites = tempsites->next;
		i++;
	}

	if (tempsites != NULL && tempsites->next != NULL)
		return tempsites->next->site - tempsites->site;

	else {
		printf("dg_get_reg_length: something went wrong\n");
		return 0;
	}
}

/*  returns the total time of the tree (counting all the branches)
 *  given the region index.
 */
genid 
dg_total_tree_time (rsitenum rindex1) 
{	
	double point = dg_reg_center (rindex1);

	Node *tempnode = dg_get_head_node (rindex1);
	genid totaltime = 0;

	totaltime += dg_calc_time_in_branch (point,
					     tempnode->gen, 
					     tempnode->descendent[0]);
	totaltime += dg_calc_time_in_branch (point,
					     tempnode->gen, 
					     tempnode->descendent[1]);

	return totaltime;
}


Node * 
dg_recurse_head_node (Node* /* nodeptr */, double /* loc */);

Node * 
dg_recurse_head_node (Node* nodeptr, double loc)
{
	int test0, test1;

	if (nodeptr->descendent[0] == NULL)
		return nodeptr;

	else if (nodeptr->descendent[1] == NULL)
		return dg_recurse_head_node (nodeptr->descendent[0], loc);
	
	else {
		test0 = seg_contains(nodeptr->descendent[0]->segs, loc); 
		test1 = seg_contains(nodeptr->descendent[1]->segs, loc); 
		if (test0 == 1 && test1 == 0)
			return dg_recurse_head_node(nodeptr->descendent[0], loc);
		else if (test0 == 0 && test1 == 1)
		  /* recurse on nodeptr->descendent[1]; */
			return dg_recurse_head_node(nodeptr->descendent[1], loc);
		else return nodeptr;
	}
}

double 
dg_reg_center (rsitenum rindex1) 
{
	int i = 0;
	struct sitelist *tempsites = demography_data.recombsites;

	if (rindex1 > demography_data.numsites) {
		printf("dg_reg_center: rindex too long\n");
		return 0;
	}

	while (i < rindex1 && tempsites != NULL) {
		tempsites = tempsites->next;
		i++;
	}

	if (tempsites != NULL && tempsites->next != NULL)
		return (tempsites->next->site + tempsites->site) / 2;

	else {
		printf("dg_reg_center: something went wrong\n");
		return 0;
	}

}

double 
dg_reg_begin (rsitenum rindex1) 
{
	int i = 0;
	struct sitelist *tempsites = demography_data.recombsites;

	if (rindex1 > demography_data.numsites) {
		printf("dg_reg_begin: rindex too long\n");
		return 0;
	}

	while (i < rindex1 && tempsites != NULL) {
		tempsites = tempsites->next;
		i++;
	}

	if (tempsites != NULL)
		return tempsites->site;

	else {
		printf("dg_reg_begin: something went wrong\n");
		return 0;
	}
}
/***********************************************************/


/***********************************************************/
/* UTILITIES FOR INTERNAL USE ONLY */

/* DG_EXIT
 */
void 
dg_exit(char* funct_name, char* error_string) 
{
	fprintf(stderr, "demography.c | %s: %s\n",
		funct_name, error_string);
	exit(EXIT_FAILURE);
}

void 
dg_error_nonfatal(char* funct_name, char* error_string) 
{
	fprintf(stderr, "demography.c | %s: %s\n",
		funct_name, error_string);
}

/* POP FUNCTIONS */
void 
dg_add_pop (Pop *newpop, genid gen, struct poplist **popheadRef) 
{
	struct poplist *temppoplist;

	temppoplist = malloc (sizeof (struct poplist));
	if (temppoplist == NULL)
		dg_exit("dg_add_pop", "malloc error for temppoplist");

	temppoplist->pop = newpop;
	temppoplist->next = *popheadRef;

	*popheadRef = temppoplist;
	demography_data.numpops ++;

	dg_log (CREATE_POP, gen, newpop);
}

/* GET POPS */
/* both functions return NULL if the specified pop is not to
 * be found.
 */
Pop *
dg_get_pop_by_name(popid popname) {
  return dg_get_pop_by_name_int(popname, demography_data.pops);
}

Pop * 
dg_get_pop_by_name_int (int index1, struct poplist *poplistptr){ 

	struct poplist *temp = poplistptr;

	while (temp != NULL) {
		if (temp->pop->name == index1)
			return temp->pop;
		temp = temp->next;
	}
	return NULL;

}

popid 
dg_get_pop_name_by_index (int popindex) {
  return dg_get_pop_by_index(popindex)->name;
}
int dg_get_pop_index_by_name (popid popname) {
  struct poplist *temp = demography_data.pops;
  int i=0;

  while (temp != NULL) {
    if (temp->pop->name == popname) {
      return i;
    }
    i++;
    temp = temp->next;
  }
  return -1;
}


Pop * 
dg_get_pop_by_index (int index1){ 
  return dg_get_pop_from_list(index1, demography_data.pops);
}

Pop *
dg_get_pop_from_list(int index1, poplist* poplistptr) {
  if (poplistptr == NULL) {
    return NULL;
  }
  else if (index1 == 0) {
    return poplistptr->pop;
  }
  else return dg_get_pop_from_list(index1-1, poplistptr->next);
}


struct poplist * 
dg_delete_pop_by_name (popid popname, struct poplist *temppoplist) 
{

	if (temppoplist == NULL) {
		dg_error_nonfatal("dg_delete_pop_by_name",
				 "couldn't delete specified pop\n");
		return NULL;
	}

	if (temppoplist->pop->name == popname) {
		if (DEMOG_DEBUG) printf("pop %s removed!\n", temppoplist->pop->label);
		demography_data.numpops --;
		return temppoplist->next;
	}

	temppoplist->next = dg_delete_pop_by_name (popname, temppoplist->next);
	
	return temppoplist;
}


/* Called by dg_done_coalescent */

int
dg_add_to_completepop (Node *nodeptr, Pop *popptr, double begin, double end)
{
	Node    *donenode = NULL, 
		*contnode = NULL;
	int number_of_nodes;

	number_of_nodes = node_break_off_seg (nodeptr, &donenode, 
					      &contnode, begin, end);

	if (number_of_nodes == 2) {
		pop_add_node(demography_data.completepop,
			     donenode);
		pop_remove_node(popptr, nodeptr);
		pop_add_node (popptr, contnode);		
		dg_log (DONE, nodeptr->gen, nodeptr, donenode, contnode);
		return 1;
	}		
	else if (number_of_nodes == 1) {
		pop_add_node(demography_data.completepop,
			     nodeptr);
		pop_remove_node(popptr, nodeptr);
		dg_log (DONE, nodeptr->gen, nodeptr, donenode, contnode);
	}
	
	return 0;
}

/* DG_ADD_RECOMB_SITE
 * adds a new recombination site if necessary to the list of
 * recombination sites.
 */
struct sitelist * 
dg_add_recomb_site(double site, struct sitelist *sitelistptr) 
{
  struct sitelist *newsitelist, *thislistptr, *lastlistptr;

  /* 1. check if site belongs at the top of the list.
   * 2. loop through list.
   * 3. check if site already exists in the list.
   * 4. add new site.
   */
  /* Modify: eliminate recursion (which blows up if the stack gets too deep). sfs */

  if (sitelistptr == NULL) {
    newsitelist = malloc (sizeof (*newsitelist));
    if (newsitelist == NULL)
      dg_exit("dg_add_recomb_site",
              "malloc error for newsitelist");

    newsitelist->site = site;
    newsitelist->next = sitelistptr;
    newsitelist->nnode = demography_data.totmembers;
    demography_data.numsites++;
    return newsitelist;
  }

  thislistptr = sitelistptr->next;
  lastlistptr = sitelistptr;

  while (thislistptr != NULL) {
    if (thislistptr->site == site) {return sitelistptr;}
    if (thislistptr->site > site) {
      newsitelist = malloc (sizeof (*newsitelist));
      if (newsitelist == NULL) {dg_exit("dg_add_recomb_site", "malloc error for newsitelist");}
      newsitelist->site = site;
      newsitelist->next = thislistptr;
      newsitelist->nnode = lastlistptr->nnode;
      lastlistptr->next = newsitelist;
      demography_data.numsites++;
      return sitelistptr;
    }
    lastlistptr = thislistptr;
    thislistptr = thislistptr->next;
  }
  newsitelist = malloc (sizeof (*newsitelist));
  if (newsitelist == NULL) {dg_exit("dg_add_recomb_site", "malloc error for newsitelist");}
  newsitelist->site = site;
  newsitelist->next = NULL;
  newsitelist->nnode = 0;
  lastlistptr->next = newsitelist;
  demography_data.numsites++;
  return sitelistptr;
}

/* MUTATION */

Node * 
dg_get_head_node (rsitenum regionnum) 
{
	Pop *popptr;
	int i;
	Node * nodeptr;
	double loc;

	/* since there's only one population.. */
	popptr = demography_data.completepop; /* dg_get_pop_by_name_int(0, demography_data.pops); */

	loc = dg_reg_center (regionnum);

	for (i = 0; i < popptr->members.nummembers; i++) {
		nodeptr = pop_get_node (i, &(popptr->members));
		if (seg_contains (nodeptr->segs, loc)) {
		  /* need to find top node  */
			return dg_recurse_head_node(nodeptr, loc);
		}
	}
	printf("head node not found\n");
	return NULL;
}

/*  used by dg_total_tree_time to calculate the overall time in
 *  all the branches of a tree. only include a branch if it contains
 *  the point 'point'.
 */
genid 
dg_calc_time_in_branch (double point, genid parenttime, Node* nodeptr) 
{
	genid time;

	if (nodeptr == NULL)
		return 0;
	
	if (! seg_contains(nodeptr->segs, point))
		return 0;
	
	time = parenttime - nodeptr->gen;	  
	time += dg_calc_time_in_branch (point, nodeptr->gen, 
					nodeptr->descendent[0]);
	time += dg_calc_time_in_branch (point, nodeptr->gen, 
					nodeptr->descendent[1]);
	return time;
}


/***********************************************************/



